// Night Eye Shader Effect for SDR

// ****** Original Bethesda Parameter ******
/// <summary>?</summary>
/// <type>Point4D</type>
/// <minValue>0.0,0.0,0.0,0.0</minValue>
/// <maxValue>1.0,1.0,1.0,1.0</maxValue>
/// <defaultValue>1.0,1.0,1.0,1.0</defaultValue>
float4 SpellInput : register(c1);

// ****** SDR Parameters ******
// Light Level [0 to 100]: Amount of light hitting the player.  Changes dynamically in game.
/// <summary>Light Level Factor| Brightness Factor | Color Factor | Tunnel Factor</summary>
/// <type>Point4D</type>
/// <minValue>0.0,0.0,0.0,0.0</minValue>
/// <maxValue>100.0,0.0,0.0,0.0</maxValue>
/// <defaultValue>100.0,0.0,0.0,0.0</defaultValue>
float4 TESR_SDR_NIGHTEYE_LightLevel : register(c4);

// Brightness Factor [0 to 100]: This determines how much the night-eye effect brightens the image.  At 0, there is no brightness increase.
// Color Mix Factor [0 to 100]: This determines how much of the image's original colors mixes with the night eye effect colors.  0 leans towards the night eye color scheme.  100 leans towards the original input color.  The color mix factor will also be dynamically affected by the light level hitting the player.
// Tunnel Factor [0 to 100]: Applies a tunnel vision effect, scaling from 0 (no tunnel) to 100 (very tunneled, about 50% visability).  This effect is then modified by the amount of light hitting the player.  A light level of 0 = 0% of the effect is applied, and a light level of 100 = 100% of the effect is applied.
/// <summary>Brightness Factor | Color Factor | Tunnel Factor</summary>
/// <type>Point4D</type>
/// <minValue>0.0,0.0,0.0,0.0</minValue>
/// <maxValue>100.0,100.0,100.0,0.0</maxValue>
/// <defaultValue>45.0,30.0,45.0,0.0</defaultValue>
float4 TESR_SDR_NIGHTEYE_Factors : register(c5);

// Color Balance
// These settings [0 to 100 each] determine the RGB color balance of the nighteye effect that is applied to the input color.
/// <summary>Balance of Red, Blue, Green</summary>
/// <type>Point4D</type>
/// <minValue>0.0,0.0,0.0,0.0</minValue>
/// <maxValue>100.0,100.0,100.0,0.0</maxValue>
/// <defaultValue>40.0,36.0,40.0,0.0</defaultValue>
float4 TESR_SDR_NIGHTEYE_Balance : register(c6);

// Color Saturation
// These settings [0 to 100 each] determine the RGB saturation of the nighteye effect that is applied to the input color.
/// <summary>Saturation levels of Red, Blue, Green.  0 = completely desaturated </summary>
/// <type>Point4D</type>
/// <minValue>0.0,0.0,0.0,0.0</minValue>
/// <maxValue>100.0,100.0,100.0,0.0</maxValue>
/// <defaultValue>19.0,42.0,42.0,0.0</defaultValue>
float4 TESR_SDR_NIGHTEYE_Saturation : register(c7);

// *** 2D Texture ***
sampler2D ScreenSpace : register(s0);

// Registers:
//
//   Name         Reg   Size
//   ------------ ----- ----
//   SpellInput   const_1       1
//   ScreenSpace  texture_0       1
//

// Structures:

struct VS_OUTPUT {
    float4 ScreenOffset : TEXCOORD0;
};

struct PS_OUTPUT {
    float4 Color : COLOR0;
};

// ****** Code ******

PS_OUTPUT main(VS_OUTPUT IN) {
    PS_OUTPUT OUT;

// ** Original Nighteye Code **
//    const int4 const_0 = {1, 0, 0, -1};
//    const float4 const_2 = {1, 1, 1, 0.5};
//    const float4 const_3 = {0.21, 0.5, 0.78, 0};

//    float q1;
//    float4 r0;
//    float4 r1;

//    IN.ScreenOffset.xyzw = tex2D(ScreenSpace, IN.ScreenOffset.xy);
    
//    r1.xyzw = dot(const_0.xyz, SpellInput.xyz);
//    r0.w = 1;
//    q1 = 0.5 - (((r1.w - 1) * (r1.w - 1)) * 0.5);
//    r0.xyz = lerp(IN.ScreenOffset.xyz, dot(const_2.xyz, IN.ScreenOffset.xyz) * const_3.xyz, q1 > 0.5 ? 0 : 1);
//    OUT.Color.xyzw = r0.xyzw;


// ** SDR NightEye Code **
  float4 ColorInput = tex2D( ScreenSpace, IN.ScreenOffset );

// Adjust brightness
  float LLAdj = TESR_SDR_NIGHTEYE_LightLevel.x * (100 - TESR_SDR_NIGHTEYE_Factors.x) * .0001;
  float3 colorBright = ColorInput.rgb;
  colorBright.rgb *= 1 + ((2 - (1.875 * LLAdj) - (pow(LLAdj,2) * .125)) * (1 - colorBright.rgb));

// Apply primary color balance settings
  float3 balance = ((TESR_SDR_NIGHTEYE_Balance.rgb * 2) - 100 ) * .0025;
  float3 colorBalanced = colorBright;
  colorBalanced.r += balance.r - (balance.g + balance.b)*.5;
  colorBalanced.g += balance.g - (balance.b + balance.r)*.5;
  colorBalanced.b += balance.b - (balance.r + balance.g)*.5;
  colorBalanced = saturate(colorBalanced);

// Desaturate
  float3 grayscaleCoef = float3(0.299,0.587,0.114);
  float3 colorGrayscaled = dot(grayscaleCoef,colorBalanced);
  float3 desaturation = (100 - TESR_SDR_NIGHTEYE_Saturation.rgb) *.01;
  float3 colorDesaturated = lerp(colorBalanced,colorGrayscaled,desaturation);

// Mix original with current color levels
  float3  colorMixed = lerp(colorDesaturated,colorBright,TESR_SDR_NIGHTEYE_LightLevel.x * TESR_SDR_NIGHTEYE_Factors.y * .0001);

// Apply tunnel
  float3 tunnelSpace = colorMixed *sin(IN.ScreenOffset.x * 3.14) * sin(IN.ScreenOffset.y * 3.14);
  float3 tunnelColor = lerp(tunnelSpace,colorMixed,1 - (TESR_SDR_NIGHTEYE_LightLevel.x * TESR_SDR_NIGHTEYE_Factors.z * .0002));

  OUT.Color.xyzw = float4(tunnelColor.r,tunnelColor.g,tunnelColor.b,ColorInput.a);

// Return Result
  
    return OUT;
};
