// Blindness Shader Effect for SDR

/// <summary>x = Blind Value of the Player.  Scale of 0 to 100</summary>
/// <type>Point4D</type>
/// <minValue>0.0,0.0,0.0,0.0</minValue>
/// <maxValue>100.0,0.0,0.0,0.0</maxValue>
/// <defaultValue>50.0,0.0,0.0,0.0</defaultValue>
float4 TESR_SDR_Blindness_Value : register(c1);

/// <summary>x = Severity of blindness effect</summary>
/// <type>Point4D</type>
/// <minValue>0.0,0.0,0.0,0.0</minValue>
/// <maxValue>100.0,0.0,0.0,0.0</maxValue>
/// <defaultValue>50.0,0.0,0.0,0.0</defaultValue>
float4 TESR_SDR_Effect_Severity : register(c2);

sampler2D TESR_RenderedBuffer : register(s0) = sampler_state { ADDRESSU = CLAMP; ADDRESSV = CLAMP; MAGFILTER = LINEAR; MINFILTER = LINEAR; MIPFILTER = LINEAR; };

static const float2 poisson[12] = 
{
  float2(-0.326212f, -0.40581f),
  float2(-0.840144f, -0.07358f),
  float2(-0.695914f, 0.457137f),
  float2(-0.203345f, 0.620716f),
  float2(0.96234f, -0.194983f),
  float2(0.473434f, -0.480026f),
  float2(0.519456f, 0.767022f),
  float2(0.185461f, -0.893124f),
  float2(0.507431f, 0.064425f),
  float2(0.89642f, 0.412458f),
  float2(-0.32194f, -0.932615f),
  float2(-0.791559f, -0.59771f)
};

static const double coefs[6] =
{
  double(0.737301398686f),
  double(6.47178645513f),
  double(-0.296262050295f),
  double(0.0064855132576f),
  double(-0.0000662787581633f),
  double(0.000000255754862603f)
};

struct VSOUT
{
  float4 vertPos : POSITION;
  float2 UVCoord : TEXCOORD0;
};
 
struct VSIN
{
  float4 vertPos : POSITION0;
  float2 UVCoord : TEXCOORD0;
};
 
VSOUT FrameVS(VSIN IN)
{
  VSOUT OUT = (VSOUT)0.0f;
  OUT.vertPos = IN.vertPos;
  OUT.UVCoord = IN.UVCoord;
  return OUT;
}

float4 main(VSOUT IN) : COLOR0
{
  // Capture Passed Blindness values
  float	BlindPercent = TESR_SDR_Blindness_Value.x * .01;
  float	Severity = 90 + (TESR_SDR_Effect_Severity.x * 0.1);
  float BlindFactor = BlindPercent * .01;
 
  // Blur Image
  float4 ImageIn = tex2D( TESR_RenderedBuffer, IN.UVCoord );
  float2 Center = float2(0.5,0.5);
  float Scale = 1 - BlindFactor;;
  float2 coord = ((IN.UVCoord.xy - Center) * Scale) + Center;
  float taps = 12.0f;
  for(int tap = 0; tap < taps; tap++){
    ImageIn += tex2D(TESR_RenderedBuffer, (coord + (poisson[tap] * BlindFactor)));
  }
  ImageIn /= taps + 1;
  
  // Blind Spot
  float4 BlindSpot = float4(ImageIn.rgb * -sin(IN.UVCoord.x * 3.14) * sin(IN.UVCoord.y * 3.14), ImageIn.a);

  // Blind Spot Percent
  float BlindSpotPercent = coefs[0];
  for(int coef = 1; coef < 6; coef++){
    BlindSpotPercent += coefs[coef] * pow(TESR_SDR_Blindness_Value.x,coef);
  }
  BlindSpotPercent = saturate(BlindSpotPercent * Severity * .0001);
    
  // Return Interpolation of the two
  return lerp(ImageIn,BlindSpot,BlindSpotPercent);
}

technique
{
  pass
  {
  VertexShader = compile vs_3_0 FrameVS();
  PixelShader = compile ps_3_0 main();
  }
}